#include <bootstrap.h>
#include <omap1510.h>
#include <dma.h>

dma_channel_t channels[8];
	
void omap_dma_irq(int ch){
	uint16_t status = 0;

	switch(ch){
	case 0:
		status = _reg16(OMAP_DMA_CSR(0));
		if(!status) {
			ch = 6;
			status = _reg16(OMAP_DMA_CSR(6));
		}
		break;
	case 1:
		status = _reg16(OMAP_DMA_CSR(1));
                if(!status) {
                        ch = 7;
                        status = _reg16(OMAP_DMA_CSR(7));
                }
                break;
	case 2:
		status = _reg16(OMAP_DMA_CSR(2));
                if(!status) {
                        ch = 8;
                        status = _reg16(OMAP_DMA_CSR(8));
                }
                break;
	default:
		status = _reg16(OMAP_DMA_CSR(ch));
		break;
	}

	if (status & OMAP_DMA_TOUT_IRQ)
                printf("DMA timeout\n");
        if (status & OMAP_DMA_DROP_IRQ)
                printf("DMA synchronization event drop occurred\n");

	// Execute the channel irq handler
	channels[ch].handler(&channels[ch], status);
}

void dma_set_transfer_params(int lch, 
			     int data_type, 
			     int elem_count,
                             int frame_count, 
			     int sync_mode)
{
        u16 w;

        w = _reg16(OMAP_DMA_CSDP(lch));
        w &= ~0x03;
        w |= data_type;
       _reg16(OMAP_DMA_CSDP(lch)) = w;

        w = _reg16(OMAP_DMA_CCR(lch));
        w &= ~(1 << 5);
        if (sync_mode == OMAP_DMA_SYNC_FRAME)
                w |= 1 << 5;
        _reg16(OMAP_DMA_CCR(lch)) = w;

        w = _reg16(OMAP_DMA_CCR2(lch));
        w &= ~(1 << 2);
        if (sync_mode == OMAP_DMA_SYNC_BLOCK)
                w |= 1 << 2;
        _reg16(OMAP_DMA_CCR2(lch)) = w;

        _reg16(OMAP_DMA_CEN(lch)) = elem_count;
        _reg16(OMAP_DMA_CFN(lch)) = frame_count;
}


void  dma_ch_set_src(int lch, 
		  int src_port, 
		  int src_amode,
                  uint32_t src_start){

	uint16_t w;

	w = _reg16(OMAP_DMA_CSDP(lch));
        w &= ~(0x1f << 2);
        w |= src_port << 2;
        _reg16(OMAP_DMA_CSDP(lch)) = w;

        w = _reg16(OMAP_DMA_CCR(lch));
        w &= ~(0x03 << 12);
        w |= src_amode << 12;
        _reg16(OMAP_DMA_CCR(lch)) = w;

        _reg16(OMAP_DMA_CSSA_U(lch)) = (src_start >> 16);
        _reg16(OMAP_DMA_CSSA_L(lch)) = src_start;
}

void dma_ch_set_dest(int lch, 
		  int dest_port, 
		  int dest_amode,
                  uint32_t dest_start)
{
        u16 w;

        w = _reg16(OMAP_DMA_CSDP(lch));
        w &= ~(0x1f << 9);
        w |= dest_port << 9;
        _reg16(OMAP_DMA_CSDP(lch)) = w;

        w = _reg16(OMAP_DMA_CCR(lch));
        w &= ~(0x03 << 14);
        w |= dest_amode << 14;
        _reg16(OMAP_DMA_CCR(lch)) = w;

        _reg16(OMAP_DMA_CDSA_U(lch)) = (dest_start >> 16);
        _reg16(OMAP_DMA_CDSA_L(lch)) = dest_start;
}


void dma_ch_free(int ch)
{
        /* Disable all DMA interrupts for the channel. */
        _reg16(OMAP_DMA_CICR(ch))  = 0;
        
	/* Make sure the DMA transfer is stopped. */
        _reg16(OMAP_DMA_CCR(ch)) = 0;
}

void dma_ch_setup(int ch, 
		  uint16_t dev, 
		  void (* handler)(dma_channel_t *,uint16_t status)){
	int i;
        u32 lch_base = DMA_BASE + ch * 0x40;

	// setup channel irq handler
	channels[ch].handler = handler;
	channels[ch].lock    = 1;
	channels[ch].id	     = ch;
	channels[ch].dev     = dev;

	// Free channel
	dma_ch_free(ch);

	// Clear channel registers
        for (i = 0; i < 0x2c; i += 2){
                _reg16((lch_base + i)) = 0;
	}

	// Set device ID
	_reg16(OMAP_DMA_CCR(ch)) = dev;
}


void dma_ch_start(int ch){
	uint16_t w =0;

	/* Read CSR to make sure it's cleared. */
        w = _reg16(OMAP_DMA_CSR(ch));

        /* Enable some nice interrupts. */
        _reg16(OMAP_DMA_CICR(ch)) = OMAP_DMA_TOUT_IRQ | 
				    OMAP_DMA_DROP_IRQ | OMAP_DMA_BLOCK_IRQ;

	w = _reg16(OMAP_DMA_CCR(ch));
        w |= OMAP_DMA_CCR_EN;
        _reg16(OMAP_DMA_CCR(ch)) = w;

	if(ch == 0 || ch == 6){
		// Enable Channels irq
		irq_unmask(DMA_CHANNEL0_IRQ);
	}
}

uint32_t dma_ch_complete(int ch) {
	// Block and wait for result
	__spin_lock(&channels[ch].lock);

	return 0;
}

void dma_disable_all() {
	u32 lch_base = 0;
	u32 ch	     = 0;
	u32 reg	     = 0;

	for(ch = 0; ch < 8; ++ch){
		dma_ch_free(ch);

		lch_base = DMA_BASE + ch * 0x40;
		for (reg = 0; reg < 0x2c; reg += 2){
                	_reg16((lch_base + reg)) = 0;
        	}
	}
}
