#include <config.h>
#include <bootstrap.h>
#include <asm/domain.h>
#include <asm/pgtable.h>
#include <asm/system.h>


region_t regions[] = {
	/* Map IO registers */
	{0xFFFB0000, 0xFFFB0000,4, PTE_TYPE_LARGE, L1_PTE_COARSE},

	/* Map pages for interrupt vector */
	{0xFFFF0000, 0x13F10000,1, PTE_TYPE_LARGE, L1_PTE_COARSE},

	/* Map what ever left in the physical memory */
	{0x00100000, 0x10000000,62, PTE_TYPE_LARGE, L1_PTE_SECTION},

	/* Map 192 SRAM */
	{0xD0000000, 0x20000000,12, PTE_TYPE_LARGE, L1_PTE_COARSE},
	
	/* Mapping last 2M to use with bootstrap */
	{0x13E00000, 0x13E00000,2, PTE_TYPE_LARGE, L1_PTE_SECTION},
};

void create_section_region(region_t *region){
	uint32_t ap	= PMD_SECT_AP_WRITE | PMD_SECT_AP_READ | PMD_BIT4;
        uint32_t paddr  = region->paddr;
        uint32_t vaddr  = region->vaddr;
        uint32_t idx    = 0;
        uint32_t l1_idx = 0;
        uint32_t *l1    = bootstrap_info.mmu_l1;

	for(idx = 0; idx < region->npages; ++idx){
		// Calculate the index of entry
		l1_idx 	    = vaddr >> 20;

		// Set entry parameters
		l1[l1_idx]  = (paddr & 0xFFF00000) | ap;
		l1[l1_idx] |= PMD_TYPE_SECT;

		// Move to the next entry
		paddr += 0x100000;
		vaddr += 0x100000;
	}
}

void create_coarse_region(uint32_t *l2, region_t *region){
	uint32_t pte	= 0;
	uint32_t ap 	= PTE_AP_RW;
	uint32_t paddr 	= region->paddr;
	uint32_t vaddr 	= region->vaddr;
	uint32_t idx	= 0;
	uint32_t l2_idx = 0;
	uint32_t l1_idx = 0;
	uint32_t *l1	= bootstrap_info.mmu_l1;
	uint32_t page	= 0;

	// Configure L1 table entry
	l1_idx	    = region->vaddr >> 20;
	l1[l1_idx]  = ((uint32_t) l2 & 0xFFFFFC00) | PMD_BIT4;
	l1[l1_idx] |= L1_PTE_COARSE;

	for(idx = 0; idx < region->npages; ++idx){
		/* Second table index */
		l2_idx = (vaddr & 0x000ff000) >> 12;

		switch(region->type){
		case PTE_TYPE_LARGE:
			pte  = (paddr & 0xffff0000);
			pte |= (ap << 10);
			pte |= (ap << 8);
			pte |= (ap << 6);
			pte |= (ap << 4);
			pte |= PTE_TYPE_LARGE;

			for(page = 0; page < 16; ++page){
				l2[l2_idx + page] = pte;
			}

			paddr += 0x10000;
			vaddr += 0x10000;
			break;
		case PTE_TYPE_SMALL:
			pte  = (paddr & 0xffff0000);
                        pte |= (ap << 10);
                        pte |= (ap << 8);
                        pte |= (ap << 6);
                        pte |= (ap << 4);
                        pte |= PTE_TYPE_SMALL;

                        l2[l2_idx + page] = pte;

			vaddr += 0x1000;
			paddr += 0x1000;
			break;
		}
	}
}

void create_regions() {
	uint32_t *l1   = NULL;
	uint32_t *l2   = NULL;
	uint32_t idx   = 0;

	l1   = bootstrap_info.mmu_l1;
        l2   = (uint32_t *) ((uint8_t *) l1 + 0x4000);
        idx  = 0;

	/* Clear L2 table */
	for(idx = 0; idx < 16896; ++idx){
		l1[idx] = 0;
	}

	/*
	 * Create mapping entry for 
	 * 1. MPU public peripherals
	 * 2. MPU/DSP shared peripherals
	 * 3. MPU private peripherals
	 */	
	create_coarse_region(l2, &regions[0]);

	/* Create mapping for interrupt vector */
	create_coarse_region(l2, &regions[1]);

	/* Map all unused pages in the physical memory */
	create_section_region(&regions[2]);

	/* Create SRAM mapping */
	l2 += 256;
	create_coarse_region(l2,&regions[3]);

	/* Create mapping for bootstrap */
	create_section_region(&regions[4]);

	/* Setup malloc information */
        bootstrap_info.mm_start   = 0x00100000;
        bootstrap_info.mm_end     = 0x00100000 + (62 * 1024 * 1024);
}

void *__pg_alloc(uint32_t flag, size_t size){
	uint32_t new = 0;
	uint32_t old = 0;
	
	if(flag){
		// Allocate  from the begining
		// this should be used to allocate
		// space for Linux and Initrd images
		old = bootstrap_info.mm_start;
		new = old + size;

		if(new > bootstrap_info.mm_end){
			return NULL;
		}
		
		// Shift the start address
		bootstrap_info.mm_start = new;
		new			= old;
	}else {
		// Allocate pages from the end
		new = bootstrap_info.mm_end - size;

		if(new < bootstrap_info.mm_start){
			return NULL;
		}

		// Shift the last address
		bootstrap_info.mm_end  = new;
	}

	return (void *) new;
}
