/******************************************************************************
 *  uboot_load                                                                *
 *  Copyright (C) 2005 Everett Coleman <gcc80x86@fuzzyneural.net>             *
 *                     Husam Senussi <husamsenussi@hotmail.com>               *
 *  http://fuzzyneural.net                                                    *
 *                                                                            *
 *  This program is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  This program is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with this program; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.*
 ******************************************************************************/
#include "uboot_load.h"
#include "setup.h"
#include <stdlib.h>
#include <unistd.h>
#ifdef WIN32
# include <WIN/windows.h>

extern BOOL VirtualCopy (LPVOID, LPVOID, DWORD, DWORD);
extern BOOL LockPages (LPVOID, DWORD, PDWORD, int);
extern LPVOID AllocPhysMem (DWORD, DWORD, DWORD, DWORD, PULONG);
#endif /* WIN32 */

#define PAGE_SIZE 0x10000
#define HEAP_SIZE 10
#define MACH_TYPE 563

#ifndef LOCKFLAG_WRITE
# define LOCKFLAG_WRITE 0x001
#endif /* LOCKFLAG_WRITE */
#ifndef LOCKFLAG_QUERY_ONLY
# define LOCKFLAG_QUERY_ONLY 0x002
#endif /* LOCKFLAG_QUERY_ONLY */
#ifndef LOCKFLAG_READ
# define LOCKFLAG_READ 0x004
#endif /* LOCKFLAG_READ */

static FILE *TX=0;
# define log_print(x, ...)  { \
    fprintf (x, __VA_ARGS__); \
    fflush (x); \
  }

#if 0
static uint32_t
get_phyaddress (void *addr) {
	uint32_t base   = (uint32_t) addr & 0xFFFFF000;
	uint32_t offset = (uint32_t) addr & 0x00000FFF;
		  
	// Query result
	ULONG result = 0;

	// Now query the physical address
	if (LockPages((LPVOID)base, PAGE_SIZE, &result, LOCKFLAG_QUERY_ONLY) != TRUE)
		return (uint32_t)-1;
	return result + offset;
} /* get_phyaddress */
#endif

static void *
alloc_phys (uint32_t size, uint32_t *ppa) {
	/* Allocate physical address */
	void	 *va	= NULL;
	// Reserve the first 16M for kernel and Initrd
	uint32_t limit	= 0x10000000 + (16 * 1024 * 1024);

	do {
		if ((va = AllocPhysMem((DWORD)size, PAGE_EXECUTE_READWRITE, 0, 0, (PULONG)ppa)) == 0) {
			return 0;
		}
	} while(*ppa <= limit);

	/*
	 * Reset the memory block
	 */
	memset((void *) va, 0, size);

	/*
	 * Return the virstual address.
	 */
	return va;
}

static FILE *
elf_open (const char *path) {
	FILE *fd=fopen (path, "rb");
	if (fd == 0)
		log_print (TX, "[error] fopen file://%s: %s\r\n", path, strerror (errno));
		return fd;
} /* elf_open */

static int
elf_read (FILE *fp, void *buf, size_t size) {
	size_t received=fread (buf, size, 1, fp);
	if (received)
		return received * size;
	return -1;
} /* elf_read */

static int
elf_read_page (FILE *fp, void *buf, size_t size) {
	size_t left   =size;
	size_t blksize=0;
	size_t read   =0;
	size_t prog   =size / 1024;

	uint8_t *ptr=(uint8_t *)buf;

	while (left) {
		blksize=(left > 1024) ? 1024 : left;
		if ((read=elf_read (fp, ptr, blksize)) == -1)
			return -1;
		left -= blksize;
		ptr  += blksize;
		if ((--prog % 20) == 0)
			log_print (TX, "#");
	} return size;
} /* elf_read_page */

static void *
load_bin_image (const char *file, uint8_t *page, image_t *image) {
	FILE *fp  =0;
	fpos_t pos=0;
//	uint8_t *save=0;
	if (!file || !file[0]) {
		log_print (TX, "[error] no file to load.\r\n");
		return 0;
	}

	if ((fp=elf_open (file)) == 0)
		return 0;
	if (fseek (fp, 0, SEEK_END) == -1) {
		log_print (TX, "[error] fseek: %s\r\n", strerror (errno));
		fclose (fp);
		return 0;
	}
	if (fgetpos (fp, &pos) == -1) {
		log_print (TX, "[error] fgetpos: %s\r\n", strerror (errno));
		fclose (fp);
		return 0;
	} image->size=(uint32_t)pos;
	if (fseek (fp, 0, SEEK_SET) == -1) {
		log_print (TX, "[error] fseek: %s\r\n", strerror (errno));
		fclose (fp);
		return 0;
	}

	if (!page) {
		if ((page=(uint8_t *)alloc_phys (image->size, &image->src)) == 0) {
			log_print (TX, "[error] alloc_phys.\r\n");
			fclose (fp);
			return 0;
		}
	} else {
		image->src=(uint32_t)page;
	}

	if (elf_read_page (fp, page, image->size) == -1) {
		log_print (TX, "\r\n[error] elf_read_page: %s\r\n", strerror (errno));
		fclose (fp);
		return 0;
	} log_print (TX, " DONE\r\n");
	fclose (fp);
	return page;
} /* load_bin_image */

static void
setup_tags (void *head, uboot_t *o, image_t *initrd) {
	struct tag	*tag = (struct tag *) head;
	
	/* start tags */
	tag->hdr.tag			= ATAG_CORE;
	tag->hdr.size			= tag_size(tag_core);
	tag->u.core.flags		= 0;
	tag->u.core.pagesize	= 0x00001000;
	tag->u.core.rootdev		= 0x0000;
	tag = tag_next(tag);

	/* command line */
	if (o->command[0]) {
		tag->hdr.tag = ATAG_CMDLINE;
		/* Add 1 for the \0 and 3 more to round up the the shift */
		tag->hdr.size = (strlen(o->command) + 1 + sizeof(struct tag_header) + 3) >> 2;
		strcpy(tag->u.cmdline.cmdline, o->command);
		tag			= tag_next(tag);
	}

	/* RAM */
	tag->hdr.tag		= ATAG_MEM;
	tag->hdr.size		= tag_size(tag_mem32);
	tag->u.mem.size		= 64 * 1024 * 1024;
	tag->u.mem.start	= 0x10000000;
	tag = tag_next(tag);

	/* initrd */
	if (initrd->dst && initrd->size) {
		tag->hdr.tag		= ATAG_INITRD2;
		tag->hdr.size		= tag_size(tag_initrd);
		tag->u.initrd.start = initrd->dst;
		tag->u.initrd.size	= initrd->size;
		tag = tag_next(tag);
	}

#if USE_VIDEO
	tag->hdr.tag					= ATAG_VIDEOTEXT;
	tag->hdr.size					= tag_size(tag_videotext);
	tag->u.videotext.video_lines	= 40;
	tag->u.videotext.video_cols		= 48;
	tag = tag_next(tag);
#endif

	/* no more tags */
	tag->hdr.tag	= ATAG_NONE;
	tag->hdr.size	= 0;
	tag = tag_next(tag);
} /* setup_tags */


#define ROUND_64K(size) ((size + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1))
#define ROUND_4B(size) ((size + 3) & ~3)

extern void bootstrap_end (void);
extern void bootstrap_start (void);
extern void boot (uint32_t, uint32_t);

static uint32_t
setup_bootstrap (FILE *tx, uboot_t *o, image_t *kernel, image_t *initrd, uint32_t *paramsPA) {
	uint32_t phyaddr=0;
	uint32_t psize  =ROUND_4B (sizeof (bootstrap_param_t));
	uint32_t tsize  =ROUND_4B (sizeof (struct tag) * 6);
	uint32_t size   =(uint32_t)bootstrap_end - (uint32_t)bootstrap_start;
	uint32_t round  =0;
	void *va =0;
	void *tag=0;
	uint8_t *secboot=0;
	bootstrap_param_t *params=0;
	image_t loader={0, };

	round=ROUND_64K ((size + psize + tsize + (o->heap * 1024)));
	if ((va=alloc_phys (round, &phyaddr)) == 0) {
		log_print (TX, "[error] alloc_phys.\r\n");
		return 0;
	}
	memcpy (va, bootstrap_start, size);
	params= (bootstrap_param_t *)((uint8_t *)va + size);
	tag   = (void *)((uint8_t *)params + psize);

	setup_tags (tag, o, initrd);

	params->initrd_dst =initrd->dst;
	params->initrd_src =initrd->src;
	params->initrd_size=initrd->size;

	params->kernel_dst =kernel->dst;
	params->kernel_src =kernel->src;
	params->kernel_size=kernel->size;

	params->tags_dst   =DEST_TAGS;
	params->tags_src   =phyaddr+size+psize;
	params->tags_size  =tsize;

	params->type=o->mach;
	*paramsPA=phyaddr+size;

	secboot=(uint8_t *)va + 4 * 1024;
	if (o->loader[0]) {
		log_print (tx, "load: loader...\r\n");
		if (load_bin_image (cfg_fixfile (o, o->loader), secboot, &loader)) {
			 log_print (tx, "  loader: src=0x%08x dst=0x%08x sz=%u\r\n", loader.src, loader.dst, loader.size);
			 params->bootstrap=phyaddr + 4 * 1024;
		} else {
			log_print (tx, "[warning] loader failed to load: continuing without it.\r\n");
			loader.src=0;
			loader.size=0;
			params->bootstrap=0;

			 if (!initrd->size && !kernel->size)
				 return 0;
		}
	} /* if (o->loader[0]) */

	log_print (tx, "\r\noptions:\r\n");
	log_print (tx, "  loader: src=0x%08x dst=0x%08x sz=%u\r\n", loader.src, loader.dst, loader.size);
	log_print (tx, "  kernel: src=0x%08x dst=0x%08x sz=%u\r\n", kernel->src, kernel->dst, kernel->size);
	log_print (tx, "  initrd: src=0x%08x dst=0x%08x sz=%u\r\n", initrd->src, initrd->dst, initrd->size);
	log_print (tx, "  tags:   src=0x%08x dst=0x%08x sz=%u\r\n", params->tags_src, params->tags_dst, params->tags_size);
	log_print (tx, "  params: 0x%08x\r\n\r\n", *paramsPA);

	return phyaddr;
} /* setup_bootstrap */

extern void asm_testx (uint32_t *);

void
bload_load (FILE *tx, uboot_t *o) {
	uint32_t pa         =0;
	uint32_t paramsPA   =0;

	image_t kernel={0, DEST_KERNEL, 0};
	image_t initrd={0, DEST_INITRD, 0};

	if (!tx && ((tx=fopen ("\\Storage Card\\udebug.txt", "w+")) == 0))
		return;
	asm_testx (&pa);
	if (pa == 55) {
		log_print (tx, "asm_testx: PASS.\r\n");
	} else {
		log_print (tx, "asm_testx: FAIL.\r\n");
		return;
	} pa=0;

	TX=tx;
	if (!o->heap)
		o->heap=HEAP_SIZE;
	if (!o->mach)
		o->mach=MACH_TYPE;
	log_print (tx, "bload...\r\n");
	log_print (tx, "  loader: %s\r\n", o->loader);
	log_print (tx, "  kernel: %s\r\n", o->kernel);
	log_print (tx, "  initrd: %s\r\n", o->initrd);
	log_print (tx, "  heap:   %d\r\n", o->heap);
	log_print (tx, "  mach:   %d\r\n", o->mach);

	if (!o->loader[0] && !o->kernel[0] && !o->initrd) {
		log_print (tx, "nothing to boot!\r\n");
		return;
	}

	if (o->kernel[0]) {
		log_print (tx, "load: kernel...\r\n");
		if (load_bin_image (cfg_fixfile (o, o->kernel), 0, &kernel)) {
			 log_print (tx, "  kernel: src=0x%08x dst=0x%08x sz=%u\r\n", kernel.src, kernel.dst, kernel.size);
		} else {
			log_print (tx, "[warning] kernel failed to load: continuing without it.\r\n");
			kernel.src=0;
			kernel.size=0;
		}
	} /* if (o->kernel[0]) */

	if (o->initrd[0]) {
		log_print (tx, "load: initrd...\r\n");
		if (load_bin_image (cfg_fixfile (o, o->initrd), 0, &initrd)) {
			 log_print (tx, "  initrd: src=0x%08x dst=0x%08x sz=%u\r\n", initrd.src, initrd.dst, initrd.size);
		} else {
			log_print (tx, "[warning] initrd failed to load: continuing without it.\r\n");
			initrd.src=0;
			initrd.size=0;
		}
	} /* if (o->initrd[0]) */


	if ((pa=setup_bootstrap (tx, o, &kernel, &initrd, &paramsPA)) == 0) {
		log_print (tx, "[error] failed to relocate bootstrap.\r\n");
		return;
	}

	log_print (tx, "bye cruel world............\r\n");
	fflush (tx);
	Sleep (1000);
	fflush (tx);
	fflush (tx);



//	log_print (tx, "pa=%08x paramsPA=%08x\r\n", pa, paramsPA);
	fclose (tx);
	boot (pa, paramsPA);
} /* bload_load */
