/******************************************************************************
 *  uboot_load                                                                *
 *  Copyright (C) 2005 Everett Coleman <gcc80x86@fuzzyneural.net>             *
 *  http://fuzzyneural.net                                                    *
 *                                                                            *
 *  This program is free software; you can redistribute it and/or modify      *
 *  it under the terms of the GNU General Public License as published by      *
 *  the Free Software Foundation; either version 2 of the License, or         *
 *  (at your option) any later version.                                       *
 *                                                                            *
 *  This program is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 *  GNU General Public License for more details.                              *
 *                                                                            *
 *  You should have received a copy of the GNU General Public License         *
 *  along with this program; if not, write to the Free Software               *
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.*
 ******************************************************************************/
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>
//#include <sys/time.h>
#include "uboot_load.h"
#include <winsock.h>

/**************************************
 * Function Prototypes                */
void memPhysReset (void);
int wp_count (void);
int wp_clear (FILE *, const char *);
void wp_check (FILE *);
int wp_list (FILE *);
/**************************************
 * Set  Structures/Prototypes         */
struct set_t {
#define STYPE_STRING 0
#define STYPE_UINT32 1
  const char  s_type;
  const char *s_variable;
  void       *s_value;
  size_t      s_size;
  const char *s_help;
};

static struct set_t sets[]={
  {STYPE_STRING, "path",    0, 0, "Default path for files."},
  {STYPE_STRING, "kernel",  0, 0, "Kernel Image."},
  {STYPE_STRING, "initrd",  0, 0, "Optional Initrd."},
  {STYPE_STRING, "command", 0, 0, "Kernel command options."},
  {STYPE_UINT32, "mach",    0, 0, "Mach Type: ${LINUX_SOURCE}/arch/arm/tools/mach-types"},
  {STYPE_STRING, "loader",  0, 0, "Boot loader image."},
  {STYPE_UINT32, "heap",    0, 0, "Heap size."},

  {0, 0, 0, 0}
};
/**************************************
 * Utility Functions                  */
char *
su_eatlws (char **string) {
  if (!string || !*string || !*(*string))
    return 0;
  while (*(*string) && isspace (*(*string)))
    *(*string)++;
  return *string;
} /* su_eatlws */

char *
su_eattws (char **string) {
  if (!string || !*string || !*(*string))
    return 0;
  while (*(*string) && isspace ((*string)[strlen ((*string))-1]))
    (*string)[strlen ((*string))-1]=0;
  return *string;
} /* su_eattws */

char *
su_pstring (char **string) {
  char *s=*string;
  char q=0;

  if (!string || !*string || !*(*string))
    return 0;

  while (*(*string)) {
    if (!q && isspace (*(*string))) {
      (*string)[0]=0;
      *(*string)++;
      break;
    } else if ((*(*string) == '\\') && (*string)[1]) {
      *(*string)++;
    } else if (*(*string) == '"') {
      q=(q+1)%2;
    } *(*string)++;
  }

  if ((*s == '"') && (s[strlen (s)-1] == '"'))
    *s++, s[strlen (s)-1]=0;

  su_eatlws (string);
  return s;
} /* su_pstring */

uint32_t
su_puint32 (char **string) {
  uint32_t s=0;
  if (!string || !(*string) || !*(*string))
    return 0;

  s=strtoul (*string, string, 0);
  su_eatlws (string);
  return s;
} /* su_puint32 */
/**************************************/

int cmd_process (uboot_t *, FILE *, FILE *, char *);

void
cmd_print (FILE *fd, const char *format, ...) {
  if (!fd)
    return;
  if (format) {
    va_list ap;
    va_start (ap, format);
    vfprintf (fd, format, ap);
    va_end (ap);
  }
  fflush (fd);
} /* cmd_print */


int
cmd_loop (uboot_t *o, FILE *rx, FILE *tx) {
  char done=0;
	struct fd_set rfd;
	struct fd_set efd;
	int max_fd=0;
  if (!o || !rx)
    return errno=EINVAL;

	if (tx) {
		stdout=tx;
		stderr=tx;
	}

  sets[0].s_value=(void *)o->directory, sets[0].s_size=sizeof (o->directory);
  sets[1].s_value=(void *)o->kernel,    sets[1].s_size=sizeof (o->kernel);
  sets[2].s_value=(void *)o->initrd,    sets[2].s_size=sizeof (o->initrd);
  sets[3].s_value=(void *)o->command,   sets[3].s_size=sizeof (o->command);
  sets[4].s_value=(void *)&o->mach,     sets[4].s_size=sizeof (o->mach);
  sets[5].s_value=(void *)&o->loader,   sets[5].s_size=sizeof (o->loader);
  sets[6].s_value=(void *)&o->heap,     sets[6].s_size=sizeof (o->heap);

  cmd_print (tx, "%s version %s, Copyright (C) 2005 Everett Coleman II\r\n", PACKAGE, VERSION);
  cmd_print (tx, "%s comes with ABSOLUTELY NO WARRANTY; for details type `help warranty'\r\n", PACKAGE);
  cmd_print (tx, "This is free software, and you are welcome to redistribute it\r\n");
  cmd_print (tx, "under certain conditions; type `help conditions' for details.\r\n\r\n");
	cmd_print (tx, "'load' code by Husam Senussi <husamsenussi@hotmail.com>, ported by me.\r\n\r\n");

	memPhysReset ();

	max_fd=fileno (rx);
  while (!done) {
    char xbuf[1024]={0, };
    char *buf=0;

    cmd_print (tx, "oboot# "); // prompt
		for (; tx;) {
			struct timeval tv={1, 0};
			int x=0;

			FD_ZERO (&rfd);
			FD_ZERO (&efd);
			FD_SET (max_fd, &rfd);
			FD_SET (max_fd, &efd);

			x=select (max_fd+1, &rfd, 0, &efd, &tv);
			if (feof (rx) || (x < 0))
				goto break_loop;

			if (x && FD_ISSET (max_fd, &efd))
				break;
			if (x && FD_ISSET (max_fd, &rfd))
				break;
			if (wp_count ())
				wp_check (tx);
		} /* while (1) */
    buf=fgets (xbuf, sizeof (xbuf), rx);
    if (!buf && feof (rx))
      break;

    su_eatlws (&buf);
    su_eattws (&buf);
    if (!buf[0] || (buf[0] == '#'))
      continue;

    done=cmd_process (o, rx, tx, buf);
  } /* while (!done) */
break_loop:
  cmd_print (tx, "bye.\r\n");
	memPhysReset ();
  return 0;
} /* cmd_loop */

int
cmd_file (uboot_t *o, const char *fx) {
  FILE *fd=0;

  if (!o || !fx)
    return errno=EINVAL;
  if (access (fx, R_OK))
    return 1;

  if ((fd=fopen (fx, "r")) == 0)
    return 1;

  if (cmd_loop (o, fd, 0)) {
    int ex=errno;
    fclose (fd);
    return errno=ex;
  } fclose (fd);
  return 0;
} /* cmd_file */

/**************************************
 * Command Structures/Prototypes      */
int xmd_load (uboot_t *, FILE *, FILE *, char *);
int xmd_help (uboot_t *, FILE *, FILE *, char *);
int xmd_quit (uboot_t *, FILE *, FILE *, char *);
int xmd_exit (uboot_t *, FILE *, FILE *, char *);
int xmd_dd   (uboot_t *, FILE *, FILE *, char *);
int xmd_d8   (uboot_t *, FILE *, FILE *, char *);
int xmd_d16  (uboot_t *, FILE *, FILE *, char *);
int xmd_d32  (uboot_t *, FILE *, FILE *, char *);
int xmd_d64  (uboot_t *, FILE *, FILE *, char *);
int xmd_set  (uboot_t *, FILE *, FILE *, char *);
int xmd_uset (uboot_t *, FILE *, FILE *, char *);
int xmd_watch8  (uboot_t *, FILE *, FILE *, char *);
int xmd_watch16 (uboot_t *, FILE *, FILE *, char *);
int xmd_watch32 (uboot_t *, FILE *, FILE *, char *);
int xmd_watch64 (uboot_t *, FILE *, FILE *, char *);
int xmd_wc      (uboot_t *, FILE *, FILE *, char *);
int xmd_wl      (uboot_t *, FILE *, FILE *, char *);

struct cmd_t {
  const char *c_command;
  int(*c_function)(uboot_t *, FILE *, FILE *, char *);
  const char *c_help;
  const char *c_opts;
};

static struct cmd_t commands[]={
  {"load",   &xmd_load, "Load BootLoader.", 0},
  {"dd",     &xmd_dd,   "Dump a memory region.", "address size file"},
	{"d8",     &xmd_d8,   "Dump a 8bit region.", "address [count [timeout]]"},
	{"d16",    &xmd_d16,  "Dump a 16bit region.", "address [count [timeout]]"},
	{"d32",    &xmd_d32,  "Dump a 32bit region.", "address [count [timeout]]"},
	{"d64",    &xmd_d64,  "Dump a 64bit region.", "address [count [timeout]]"},
  {"exit",   &xmd_exit, "Exit.", 0},
  {"help",   &xmd_help, "Print command help.", "[command]"},
  {"quit",   &xmd_quit, "Quit Session.", 0},
  {"set",    &xmd_set,  "Set Variables.", "[variable [value]]"},
  {"unset",  &xmd_uset, "Unset Variables.", "variable"},
	{"watch8", &xmd_watch8,   "Watch an 8bit region.", "string address"},
	{"watch16", &xmd_watch16, "Watch an 16bit region.", "string address"},
	{"watch32", &xmd_watch32, "Watch an 32bit region.", "string address"},
	{"watch64", &xmd_watch64, "Watch an 64bit region.", "string address"},
	{"wc",      &xmd_wc,      "Clear a watch point.",   "string"},
	{"wl",      &xmd_wl,      "List Watch points.",     ""},

  {0, 0, 0, 0}
};
/**************************************
 * Command Functions                  */
int
xmd_help (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  int i=0;

  if (args && args[0] && strlen (args)) {
    char *command=su_pstring (&args);

		if (!strcasecmp (command, "conditions")) {
			cmd_print (tx, "%s is distributed under the GPL, see file 'COPYING.gz' in source tree for more details.\r\n", PACKAGE);
			return 0;
		}

    for (i=0; commands[i].c_command; i++)
      if (!strcasecmp (command, commands[i].c_command)) {
        cmd_print (tx, "%s:\r\n%s\r\n", commands[i].c_command, commands[i].c_help);
        if (commands[i].c_opts) 
          cmd_print (tx, "\r\n%s %s\r\n", commands[i].c_command, commands[i].c_opts);
        return 0;
      }
  } /* if (command && command[0] && strlen (command)) */

  for (i=0; commands[i].c_command; i++)
    cmd_print (tx, "%-10s %s\r\n", commands[i].c_command, commands[i].c_help);
  return 0;
} /* xmd_help */

int
xmd_exit (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  int c=0;
  if (tx) {
    cmd_print (tx, "exit terminates %s: are you sure? [Y/n] ", PACKAGE);
    c=fgetc (rx);
    if (toupper (c) == 'Y') {
      set_flag (o->flags, FLAG_DONE);
      return 1;
    }
  } else {
    set_flag (o->flags, FLAG_DONE);
    return 1;
  }
  return 0;
} /* xmd_exit */

int
xmd_quit (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  return 1;
} /* xmd_quit */

int memory_dump (FILE *, uint32_t, uint32_t, const char *);

int
xmd_dd (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  uint32_t addr =su_puint32 (&args);
  uint32_t size =su_puint32 (&args);
  char *filename=su_pstring (&args);

  if (!filename || !*filename) {
    cmd_print (tx, "useage: dd offset size file\r\n");
  } else {
#ifdef DEBUG_PACKAGE
    cmd_print (tx, "[debug] address=0x%08x size=%u file=%s\r\n", addr, size, filename);
#endif /* DEBUG_PACKAGE */
		if (memory_dump (tx, addr, size, cfg_fixfile (o, filename)))
			cmd_print (tx, "[error] memory_dump\n");

  }
  return 0;
} /* xmd_dd */

void
print_set (int off, FILE *tx) {
  if (off >= sizeof (sets)/sizeof (struct set_t))
    return;
  cmd_print (tx, "%15s=", sets[off].s_variable);
  switch (sets[off].s_type) {
    case STYPE_STRING:
      cmd_print (tx, "\"%s\"", (const char *)sets[off].s_value);
      break;

    case STYPE_UINT32:
      cmd_print (tx, "%u", *(uint32_t *)sets[off].s_value);
      break;
  } /* switch (sets[off].s_type) */
  cmd_print (tx, "\r\n");
} /* print_set */

int
xmd_uset (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  int i=0;
	char *var=su_pstring (&args);
	if (!var)
		return 0;

	for (i=0; sets[i].s_variable; i++)
		if (!strcasecmp (var, sets[i].s_variable)) {
			switch (sets[i].s_type) {
				case STYPE_STRING:
					((char *)sets[i].s_value)[0]=0;
					break;

				case STYPE_UINT32:
					*(uint32_t *)sets[i].s_value=0;
					break;
			} return 0;
		}
	return 0;
} /* xmd_uset */

int
xmd_set (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  int i=0;
  char *var=su_pstring (&args);
  char *val=su_pstring (&args);


  val=(val && !*val) ? 0 : val;
  var=(var && !*var) ? 0 : var;

#ifdef DEBUG_PACKAGE
  cmd_print (tx, "[debug] set %s: %s\r\n", var, val);
#endif /* DEBUG_PACKAGE */
  if (!var && !val) {
    for (i=0; sets[i].s_variable; i++)
      print_set (i, tx);
    return 0;
  } else if (var && !val) {
    for (i=0; sets[i].s_variable; i++)
      if (!strcasecmp (var, sets[i].s_variable)) {
        cmd_print (tx, "%s:", sets[i].s_variable);
        if (sets[i].s_help)
          cmd_print (tx, "%s\r\n\r\n", sets[i].s_help);
        else 
          cmd_print (tx, "\r\n\r\n");

        print_set (i, tx);
        return 0;
      } /* if (!strcasecmp (var, sets[i].s_variable)) */
  } else if (var && val) {
    for (i=0; sets[i].s_variable; i++)
      if (!strcasecmp (var, sets[i].s_variable)) {
        switch (sets[i].s_type) {
          case STYPE_STRING:
            snprintf ((char *)sets[i].s_value, sets[i].s_size, "%s", val);
            break;

          case STYPE_UINT32:
            *(uint32_t *)sets[i].s_value=strtol (val, 0, 0);
            break;
        } /* switch (sets[i].s_type) */
        return 0;
      } /* if (!strcasecmp (var, sets[i].s_variable)) */
  }

  return 0;
} /* xmd_set */

int
xmd_load (uboot_t *o, FILE *rx, FILE *tx, char *args) {
  bload_load (tx, o);
  cmd_print (tx, "[error] boot loader did not load.\r\n");
  return 0;
} /* xmd_load */

int dump_region (FILE *, int, uint32_t, uint32_t, uint32_t);
int watch_region (FILE *, int, const char *, uint32_t);

int
xmd_d8 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	uint32_t address=su_puint32 (&args);
	uint32_t count  =su_puint32 (&args);
	uint32_t wait   =su_puint32 (&args);
#ifndef DEBUG_PACKAGE
	cmd_print (tx, "[debug] %s: address=%08X count=%u, wait=%u\r\n", __FUNCTION__, address, count, wait);
#endif
	return dump_region (tx, 8, address, count, wait);
} /* xmd_d8 */

int
xmd_d16 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	uint32_t address=su_puint32 (&args);
	uint32_t count  =su_puint32 (&args);
	uint32_t wait   =su_puint32 (&args);
#ifdef DEBUG_PACKAGE
	cmd_print (tx, "[debug] %s: address=%08X count=%u, wait=%u\r\n", __FUNCTION__, address, count, wait);
#endif
	return dump_region (tx, 16, address, count, wait);
} /* xmd_d16 */

int
xmd_d32 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	uint32_t address=su_puint32 (&args);
	uint32_t count  =su_puint32 (&args);
	uint32_t wait   =su_puint32 (&args);
#ifdef DEBUG_PACKAGE
	cmd_print (tx, "[debug] %s: address=%08X count=%u, wait=%u\r\n", __FUNCTION__, address, count, wait);
#endif
	return dump_region (tx, 32, address, count, wait);
} /* xmd_d32 */

int
xmd_d64 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	uint32_t address=su_puint32 (&args);
	uint32_t count  =su_puint32 (&args);
	uint32_t wait   =su_puint32 (&args);
#ifdef DEBUG_PACKAGE
	cmd_print (tx, "[debug] %s: address=%08X count=%u, wait=%u\r\n", __FUNCTION__, address, count, wait);
#endif
	return dump_region (tx, 64, address, count, wait);
} /* xmd_d64 */

int watch_region (FILE *, int, const char *, uint32_t);

int
xmd_watch8 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	char *string    =su_pstring (&args);
	uint32_t address=su_puint32 (&args);
	return watch_region (tx, 8, string, address);
} /* xmd_watch8 */

int
xmd_watch16 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	char *string    =su_pstring (&args);
	uint32_t address=su_puint32 (&args);
	return watch_region (tx, 16, string, address);
} /* xmd_watch16 */

int
xmd_watch32 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	char *string    =su_pstring (&args);
	uint32_t address=su_puint32 (&args);
	return watch_region (tx, 32, string, address);
} /* xmd_watch32 */

int
xmd_watch64 (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	char *string    =su_pstring (&args);
	uint32_t address=su_puint32 (&args);
	return watch_region (tx, 64, string, address);
} /* xmd_watch64 */

int
xmd_wc (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	char *string=su_pstring (&args);
	return wp_clear (tx, string);
} /* xmd_wc */

int
xmd_wl (uboot_t *o, FILE *rx, FILE *tx, char *args) {
	return wp_list (tx);
} /* xmd_wl */

/**************************************/

int
cmd_process (uboot_t *o, FILE *rx, FILE *tx, char *command_line) {
  int i=0;
  char *command=su_pstring (&command_line);

  if (!command)
    return 0;

#ifdef DEBUG_PACKAGE
  cmd_print (tx, "[debug] command=%s args=%s\r\n", command, command_line);
#endif /* DEBUG_PACKAGE */

  for (i=0; commands[i].c_command; i++)
    if (!strcasecmp (command, commands[i].c_command))
        return (*commands[i].c_function) (o, rx, tx, command_line);
  cmd_print (tx, "%s: no such command.\r\n", command);
  return 0;
} /* cmd_process */
